function X = irls(y, U, N, options)

% INPUT
% y - measurement vector (time domain, dimensions M x 1)
% U - sampling scheme with increment numbers (M x 1)
% N - length of full signal

% PARAMETRES
% options.eps ∈(0; ...)
% options.max_iter - stopping criteria by number of iterations [default max_iter = 20]
% options.norm - final p-norm set in the last iteration [default norm = 0]
% options.lambda - lambda [default = 1]

% POSSIBLE MODIFICATIONS:
% change of the norm from iteration to iteration: options.iterative = 1 [default]

% OUTPUT
% X (frequency domain, N x 1)

%%%%%%%%%%%% parameters %%%%%%%%%%%%
if isfield(options, 'eps')
    eps = options.eps;
    set_eps_flag=0;
else
    set_eps_flag=1 ;
end

if isfield(options, 'max_iter')
    max_iter = options.max_iter;
else
    max_iter = 20;
end

if isfield(options, 'norm')
    norm = options.norm;
else
    norm = 1;
end

if isfield(options, 'iterative')
    iterative = options.iterative;
else
    iterative = 0;
end

if isfield(options, 'lambda')
    lambda = options.lambda;
else
    lambda = 1;
end

%%%%%%%%%%%% initialization %%%%%%%%%%%%
k = 0;
F = ifft(eye(N)); % inverse Fourier transform matrix
A = F(U, :); % measurement matrix: x = A*y
fn = size(A, 2);
L = eye(size(A, 1));
K = size(A, 1);
X = A'*y;

if (set_eps_flag) % set regularization automatically
    x_sorted = sort(abs(X), 'descend');
    eps = x_sorted(1)/10000.;
    u = floor(K/4/log(fn/K)) + 1;
end

%%%%%%%%%%%% main loop %%%%%%%%%%%%
while k < max_iter
%     disp(horzcat('iteration', ' ', int2str(k)))
    if (set_eps_flag)
        x_sorted = sort(abs(X), 'descend');
        eps = min(eps, abs(x_sorted(u)));
    end
    if iterative
        tau = (k + 1)/max_iter*(1. - norm);
    else
        tau = 1. - norm;
    end
    W = diag(abs(X).^(1. + tau) + eps^(1. + tau));
    G2 = A*W;
    B = G2*A' + L*lambda;
    % Here, B is inverted by cholesky factorization. Works fast, but only for A (complex, symmetric).
    B(logical(eye(size(B)))) = real(diag(B));
    R = chol(B);
    F = R\(R'\y);
    X = G2'*F;
    k = k + 1;
end
end
